set more off
clear
set type double

********************************************************************************
********************************************************************************
********************************************************************************
**** STATA Guide
**** Building the SDG economy: Needs, spending, and financing for universal achievement of the Sustainable Development Goals
**** Homi Kharas & John McArthur
**** Final data and do-files compiled by Selen Özdoğan
**** October 2019
********************************************************************************
**** 4. Replicate figures and tables
********************************************************************************
********************************************************************************
********************************************************************************


********************************************************************************
**** Outline *******************************************************************
********************************************************************************
* 1. Figures
*		A . FIGURE 1 - Child mortality rate and public health spending per capita, 2015 
*		B . FIGURE 2A - Estimated SDG public spending per capita and GDP per capita, 2015 (countries with complete data across sectors)
* 		C . FIGURE 2B - Estimated SDG public spending per capita and GDP per capita, 2015 (includes countries with interpolated data)
* 		D . FIGURE 3A & 3B - Estimated total SDG public spending, 2015 and 2030, by initial income group and by sector
* 		E . FIGURE 4 - The Hockey Stick: Estimated SDG spending needs and GDP per capita, 2025
* 		F . FIGURE 5 - Estimated SDG needs gap (% of GDP) and GDP per capita, 2025
*		G . FIGURE 6 - Domestic resource mobilization per capita and GDP per capita, 2015
*		H . FIGURE 7 - Net ODA disbursement per capita and GDP per capita, 2015
* 		I . FIGURE 8 - Net ODA disbursement per capita and estimated SDG needs gap per capita, 2015
*		J . FIGURE 9 - Net official non-concessional financing per capita and GDP per capita, 2015
* 		K . FIGURE A1 - Net ODA disbursement and GDP per capita, 2015, in countries with population less than 300,000
* 		L . FIGURE A2 - Net ODA disbursement and estimated SDG needs gap per capita, 2015, in countries with population less than 300,000
* 2. Tables
* 		A . TABLE 1 - Estimated SDG public spending per capita in 2015, averages by sector and income group
* 		B . TABLE 2 - Estimated 2025 SDG spending needs, by sector and income group average
* 		C . TABLE 3 - Estimated aggregate 2025 SDG needs gap, by country income group
* 		D . TABLE 4 - Estimated contribution towards 2025 SDG needs gap from extra 1% of GDP in domestic resource mobilization effort, by initial income group
* 		E . TABLE A3 - Estimated aggregate 2025 SDG needs gap, by country income group, when spending fixed as share of GDP

* Note: All WDI data downloaded on 10/10/2019

********************************************************************************
********************************************************************************
*** 1. Figures
********************************************************************************
********************************************************************************

********************************************************************************
** A . FIGURE 1 - Child mortality rate and public health spending per capita, 2015 
********************************************************************************
use "output\merged_full_totals_cons.dta", clear
preserve
wbopendata, indicator(SH.DYN.MORT) nometadata year(2015) clear

drop if region == "Aggregates"
replace region = "EAP" if countryname == "Nauru"
drop if region == ""
rename yr2015 cmr_2015

keep countryname countrycode cmr_2015
tempfile cmr_2015
save `cmr_2015'.dta, replace 
restore

merge 1:1 countrycode using `cmr_2015'.dta
drop if _merge==2
drop _merge

twoway(scatter cmr_2015 pc_healthgdp_2015, mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log) yscale(log) xlabel(10 50 250 1000 5000, labsize(medsmall)) ylabel(10 25 50 100 150, labsize(medsmall) angle(horizontal)) ytitle("Under-5 child mortality rate, 2015" "(deaths per 1,000 live births, log scale)", size(medsmall)) xtitle("Public health spending per capita, 2015 (USD, log scale)", size(medsmall)) bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 10 Figure 1.png", as(png) replace

********************************************************************************
** B . FIGURE 2A - Estimated SDG public spending per capita and GDP per capita, 2015 (countries with complete data across sectors)
********************************************************************************

twoway(scatter pc_cntytolgdp_2015 pc_gdp_traj2015 if (count2015==7), mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log) yscale(log range(25 40000)) xlabel(500 1500 4000 12000 35000 110000, labsize(medsmall)) ylabel(100 500 2500 10000 40000, labsize(medsmall) angle(horizontal)) ytitle("Estimated SDG public spending per capita, 2015" "(USD, log scale)", size(medsmall)) xtitle("GDP per capita, 2015" "(USD, log scale)", size(medsmall)) bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 10 Figure 2A.png", as(png) replace

********************************************************************************
** C . FIGURE 2B - Estimated SDG public spending per capita and GDP per capita, 2015 (includes countries with interpolated data)
********************************************************************************

twoway(scatter a_pc_cntytolgdp_2015 pc_gdp_traj2015, mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log) yscale(log range(25 40000)) xlabel(500 1500 4000 12000 35000 110000, labsize(medsmall)) ylabel(100 500 2500 10000 40000, labsize(medsmall) angle(horizontal)) ytitle("Estimated SDG public spending per capita, 2015" "(USD, log scale)", size(medsmall)) xtitle("GDP per capita, 2015" "(USD, log scale)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 10 Figure 2B.png", as(png) replace

********************************************************************************
** D . FIGURE 3A & 3B - Estimated total SDG public spending, 2015 and 2030, by initial income group and by sector
********************************************************************************

* Creating per capita GDP growth
gen gdp_pc_growth1530 = pc_gdp_traj2030/pc_gdp_traj2015 - 1
gen gdp_pc_growth1525 = pc_gdp_traj2025/pc_gdp_traj2015 - 1

* Inflating by factor of 1.13
gen pc_spending_growth1530 = gdp_pc_growth1530 * 1.13
gen pc_spending_growth1525 = gdp_pc_growth1525 * 1.13

foreach sec in aggdp healthgdp edgdp SSgdp infgdp consgdp justicegdp{
	gen pc_`sec'_2030_alt = pc_`sec'_2015 * (1 + pc_spending_growth1530)
	gen tol_`sec'_2030_alt = pc_`sec'_2030_alt*pop2030
	
	gen a_pc_`sec'_2030_alt = a_pc_`sec'_2015 * (1 + pc_spending_growth1530)
	gen a_tol_`sec'_2030_alt = a_pc_`sec'_2030_alt*pop2030
	
	gen pc_`sec'_2025_alt = pc_`sec'_2015 * (1 + pc_spending_growth1525)
	gen tol_`sec'_2025_alt = pc_`sec'_2025_alt*pop2025
	
	gen a_pc_`sec'_2025_alt = a_pc_`sec'_2015 * (1 + pc_spending_growth1525)
	gen a_tol_`sec'_2025_alt = a_pc_`sec'_2025_alt*pop2025
}

*Generate totals
gen pc_cntytolgdp_2030_alt = pc_cntytolgdp_2015*(1 + pc_spending_growth1530)
gen cntytolgdp_2030_alt = pc_cntytolgdp_2030_alt*pop2030

gen pc_cntytolgdp_2025_alt = pc_cntytolgdp_2015*(1 + pc_spending_growth1525)
gen cntytolgdp_2025_alt = pc_cntytolgdp_2025_alt*pop2025

gen a_pc_cntytolgdp_2030_alt = a_pc_cntytolgdp_2015*(1 + pc_spending_growth1530)
gen a_cntytolgdp_2030_alt = a_pc_cntytolgdp_2030_alt*pop2030

gen a_pc_cntytolgdp_2025_alt = a_pc_cntytolgdp_2015*(1 + pc_spending_growth1525)
gen a_cntytolgdp_2025_alt = a_pc_cntytolgdp_2025_alt*pop2025

save "output\merged_full_totals_cons with alt.dta", replace

preserve
collapse (first) a_inctolSDG_2015, by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 10 Figure 3A and 3B.xlsx", firstrow(varlabels) sheet("incgroups") sheetreplace
restore

preserve
collapse (sum) a_cntytolgdp_2030_alt, by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 10 Figure 3A and 3B.xlsx", sheet("2030 IncGrp + multiplier") sheetreplace firstrow(variables)
restore

preserve
collapse (first) a_tol*_2015
export excel using "output\graphs\FINAL OCT 10 Figure 3A and 3B.xlsx", firstrow(varlabels) sheet("sectors") sheetreplace
restore

preserve
collapse (sum) a_tol_*_2030_alt
export excel using "output\graphs\FINAL OCT 10 Figure 3A and 3B.xlsx", sheet("2030 Sector + multiplier") sheetreplace firstrow(variables)
restore

********************************************************************************
** E . FIGURE 4 - The Hockey Stick: Estimated SDG spending needs and GDP per capita, 2025
********************************************************************************
use "output/needs_combine_full_cons.dta", clear

twoway(scatter pc_total_need pc_gdp_traj2025 if pc_gdp_traj2025<=12500, mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log range(200 12500)) yscale(log range(200 4000)) xlabel(500 1500 4000 12000, labsize(medsmall)) ylabel(200 500 1000 2000 4000, labsize(medsmall) angle(horizontal)) xtitle("GDP per capita, 2025 (USD 2015, log scale)", size(medsmall)) ytitle("Estimated SDG financing need per capita, 2025" "(USD 2015, log scale)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 16 Figure 4.png", as(png) replace

********************************************************************************
** F . FIGURE 5 - Estimated SDG needs gap (% of GDP) and GDP per capita, 2025
********************************************************************************
use "output/gaps.dta", clear

* Calculate gap as % of GDP
gen a_abs_gap_pct = ((pc_total_need-a_pc_cntytolgdp_2025_alt)/pc_gdp_traj2025)*100

scatter a_abs_gap_pct pc_gdp_traj2025 if pc_gdp_traj2025<25000, mlabel(countrycode) mlabsize(vsmall) msize(tiny) xscale(log) xlabel(500 1500 4000 12000 25000, labsize(medsmall)) yscale(range(-25 150)) ylabel(-25 0 25 50 75 100 125 150, labsize(medsmall) angle(horizontal)) xtitle("GDP per capita, 2025 (USD 2015, log scale)", size(medsmall)) ytitle("Estimated SDG needs gap in 2025 (% of GDP)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 16 Figure 5.png", as(png) replace

********************************************************************************
** G . FIGURE 6 - Domestic resource mobilization per capita and GDP per capita, 2015  
********************************************************************************
* Note - DRM defined as government revenue, excluding grants but including social contributions
use "output\spending plus DRM ODA.dta", clear 

twoway(scatter pc_rev_gdp_2015 pc_gdp_traj2015 if pc_gdp_traj2015<=25000, mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log range(300 25000)) yscale(log) xlabel(500 1500 4000 12000 25000, labsize(medsmall)) ylabel(100 500 2500 10000, labsize(medsmall) angle(horizontal)) ytitle("Domestic resource mobilization per capita, 2015" "(USD, log scale)", size(medsmall)) xtitle("GDP per capita, 2015 (USD, log scale)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 10 Figure 6.png", as(png) replace

********************************************************************************
** H . FIGURE 7 - Net ODA disbursement per capita and GDP per capita, 2015 
********************************************************************************

preserve
drop if pop2015<300000
twoway(scatter netoda_pc pc_gdp_traj2015 if (pc_gdp_traj2015<25000 & netoda_pc>0), mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log ) xlabel(500 1500 4000 12000 25000, labsize(medsmall)) yscale(log range(5 400)) ylabel(10 50 100 200 300 400, labsize(medsmall) angle(horizontal)) ytitle("Net ODA disbursement per capita, 2015" "(USD, log scale)", size(medsmall)) xtitle("GDP per capita, 2015 (USD, log scale)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 16 Figure 7.png", as(png) replace
restore 

********************************************************************************
** I . FIGURE 8 - Net ODA disbursement per capita and estimated SDG needs gap per capita, 2015 
********************************************************************************
merge m:m countryname using "output\merged_full_totals_cons.dta"
drop _merge
merge m:m countryname using "output\needs_combine_full_cons.dta"
drop _merge

gen a_abs_gap_pc2015 = pc_total_need2015-a_pc_cntytolgdp_2015

preserve
drop if pop2015<300000
twoway(scatter netoda_pc a_abs_gap_pc2015 if (netoda_pc>0), mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) yscale(log range(5 400)) ylabel(10 50 100 200 300 400, labsize(medsmall) angle(horizontal))  ytitle("Net ODA disbursement per capita, 2015" "(USD, log scale)", size(medsmall)) xtitle("Estimated SDG needs gap per capita, 2015 (USD)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 16 Figure 8.png", as(png) replace
restore

********************************************************************************
** J . FIGURE 9 - Net official non-concessional financing per capita and GDP per capita, 2015
********************************************************************************
preserve
wbopendata, indicator(DT.NFL.BLAT.CD; DT.NFL.MLAT.CD; DT.NFL.MLTC.CD; DT.NFL.BLTC.CD) nometadata year(2015) long clear

drop if region == "Aggregates"
replace region = "EAP" if countryname == "Nauru"
drop if region == ""

rename dt_nfl_blat_cd net_fin_bilat
rename dt_nfl_mlat_cd net_fin_multi
rename dt_nfl_mltc_cd net_concess_multi
rename dt_nfl_bltc_cd net_concess_bilat

gen net_concessional=net_concess_multi+net_concess_bilat
gen net_financial = net_fin_bilat+net_fin_multi
gen net_nc_finance_2015 = net_financial-net_concessional

keep countryname countrycode net_nc_finance_2015
tempfile net_nc_finance_2015
save `net_nc_finance_2015'.dta, replace 
restore

merge m:m countrycode using `net_nc_finance_2015'.dta
drop if _merge==2
drop _merge

gen pc_net_nc_finance_2015 = net_nc_finance_2015/pop2015
save `net_nc_finance'.dta, replace 
use `net_nc_finance'.dta, clear

twoway(scatter pc_net_nc_finance_2015 pc_gdp_traj2015 if (pc_gdp_traj2015<15000), mlabel(countrycode) mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log) xlabel(500 1500 4000 12000, labsize(medsmall)) ytitle("Net official non-concessional financing per capita, 2015" "(USD)", size(medsmall)) xtitle("GDP per capita, 2015 (USD, log scale)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 17 Figure 9.png", as(png) replace

********************************************************************************
** K . FIGURE A1 - Net ODA disbursement and GDP per capita, 2015, in countries with population less than 300,000
********************************************************************************
use "output\spending plus DRM ODA.dta", clear 

preserve
drop if pop2015>=300000
twoway(scatter netoda_pc pc_gdp_traj2015 if (pc_gdp_traj2015<25000), mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) xscale(log) xlabel(500 1500 4000 12000 25000, labsize(medsmall)) yscale(log range(1 6250)) ylabel(10 50 250 1250 6250, labsize(medsmall) angle(horizontal)) ytitle("Net ODA disbursement per capita, 2015" "(USD, log scale)", size(medsmall)) xtitle("GDP per capita, 2015 (USD, log scale)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 17 Figure A1.png", as(png) replace
restore 

********************************************************************************
** L . FIGURE A2 - Net ODA disbursement and estimated SDG needs gap per capita, 2015, in countries with population less than 300,000
********************************************************************************
merge m:m countryname using "output\merged_full_totals_cons.dta"
drop _merge
merge m:m countryname using "output\needs_combine_full_cons.dta"
drop _merge

gen a_abs_gap_pc2015 = pc_total_need2015-a_pc_cntytolgdp_2015

preserve
drop if pop2015>=300000
twoway(scatter netoda_pc a_abs_gap_pc2015, mlabel(countrycode) mlabsize(vsmall) msize(tiny)), legend(off) yscale(log range(1 6250)) ylabel(10 50 250 1250 6250, labsize(medsmall) angle(horizontal))  ytitle("Net ODA disbursement per capita, 2015" "(USD, log scale)", size(medsmall)) xtitle("Estimated SDG needs gap per capita, 2015 (USD)", size(medsmall))  bgcolor(none) graphregion(color(white))
graph export "output\graphs\FINAL OCT 16 Figure A2.png", as(png) replace
restore 

********************************************************************************
********************************************************************************
*** 2. Tables
********************************************************************************
********************************************************************************

********************************************************************************
** A . TABLE 1 - Estimated SDG public spending per capita in 2015, averages by sector and income group
********************************************************************************
use "output\merged_full_totals_cons.dta", clear

preserve
collapse (mean) a_pc_aggdp_2015 a_pc_healthgdp_2015 a_pc_edgdp_2015 a_pc_SSgdp_2015 a_pc_infgdp_2015 a_pc_consgdp_2015 a_pc_justicegdp_2015 [aweight = pop2015], by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 17 Table 1.xlsx", firstrow(varlabels) sheet("sector") sheetreplace
restore

preserve
collapse (mean) a_pc_aggdp_2015 a_pc_healthgdp_2015 a_pc_edgdp_2015 a_pc_SSgdp_2015 a_pc_infgdp_2015 a_pc_consgdp_2015 a_pc_justicegdp_2015 [aweight = pop2015]
export excel using "output\graphs\FINAL OCT 17 Table 1.xlsx", firstrow(varlabels) sheet("global") sheetreplace
restore

********************************************************************************
** B . TABLE 2 - Estimated 2025 SDG spending needs, by sector and income group average
********************************************************************************
use "output/needs_combine_full_cons.dta", clear

preserve
table incomegroup2015 [aweight = pop2025], c(mean pc_total_need)
collapse (mean) pc_total_need [aweight = pop2025], by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table 2.xlsx", firstrow(varlabels) sheet("incgroups") sheetreplace
restore 

preserve
collapse (mean) pc_ag_need1530 pc_health_need2030 pc_educ_needs pc_pgtSS_need2025 pc_energy_needs2025 pc_flood_needs2025 pc_transport_needs2025 pc_wash_need2025 pc_biodiversity_needs pc_justice_needs [aweight = pop2025], by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table 2.xlsx", firstrow(varlabels) sheet("sector") sheetreplace
restore

preserve
collapse (mean) pc_ag_need1530 pc_health_need2030 pc_educ_needs pc_pgtSS_need2025 pc_energy_needs2025 pc_flood_needs2025 pc_transport_needs2025 pc_wash_need2025 pc_biodiversity_needs pc_justice_needs [aweight = pop2025]
export excel using "output\graphs\FINAL OCT 16 Table 2.xlsx", firstrow(varlabels) sheet("all developing") sheetreplace
restore

********************************************************************************
** C . TABLE 3 - Estimated aggregate 2025 SDG needs gap, by country income group
********************************************************************************
use "output/gaps.dta", clear

preserve
table incomegroup2015, c(sum gap_total_2025_alt) format(%10.0g)
collapse (sum) gap_total_2025_alt, by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table 3.xlsx", firstrow(variables) sheetreplace sheet("incgroup total")
restore

preserve
table incomegroup2015 [aweight=pop2025], c(mean gap_pc_2025_alt) format(%10.0g)
collapse (mean) gap_pc_2025_alt [aweight=pop2025], by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table 3.xlsx", firstrow(variables) sheetreplace sheet("incgroup pc")
restore

preserve
table incomegroup2015 [aweight=pop2025], c(mean gap_pc_2025_alt) format(%10.0g)
collapse (mean) gap_pc_2025_alt [aweight=pop2025]
export excel using "output\graphs\FINAL OCT 16 Table 3.xlsx", firstrow(variables) sheetreplace sheet("avg pc")
restore

********************************************************************************
** D . TABLE 4 - Estimated contribution towards 2025 SDG needs gap from extra 1% of GDP in domestic resource mobilization effort, by initial income group
********************************************************************************

preserve
collapse (sum) gap_contribution_2025_alt, by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table 4.xlsx", firstrow(variables) sheetreplace sheet("x1.13 + 1 % gap 2025")
restore

preserve
table incomegroup2015 [aweight=pop2025], c(mean gap_pc_2025_alt_1) format(%10.0g)
collapse (mean) gap_pc_2025_alt_1 gap_pc_2025_alt [aweight=pop2025], by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table 4.xlsx", firstrow(variables) sheetreplace sheet("incgroup pc")
restore

preserve 
collapse (mean) gap_pc_2025_alt_1 gap_pc_2025_alt [aweight=pop2025]
export excel using "output\graphs\FINAL OCT 16 Table 4.xlsx", firstrow(variables) sheetreplace sheet("avg pc")
restore


********************************************************************************
** E . TABLE A3 - Estimated aggregate 2025 SDG needs gap, by country income group, when spending fixed as share of GDP
********************************************************************************

preserve
collapse (sum) gap_contribution_2025 gap_total_2025, by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table A1.xlsx", firstrow(variables) sheetreplace sheet("wout multiplier + 1 % gap 2025")
restore

preserve
table incomegroup2015 [aweight=pop2025], c(mean gap_pc_2025_1) format(%10.0g)
collapse (mean) gap_pc_2025_1 gap_pc_2025 [aweight=pop2025], by(incomegroup2015)
export excel using "output\graphs\FINAL OCT 16 Table A1.xlsx", firstrow(variables) sheetreplace sheet("incgroup pc")
restore

preserve 
collapse (mean) gap_pc_2025_1 gap_pc_2025 [aweight=pop2025]
export excel using "output\graphs\FINAL OCT 16 Table A1.xlsx", firstrow(variables) sheetreplace sheet("avg pc")
restore
